      /* ============================================================================
          GLOBAL VARIABLES & DOM ELEMENTS
      ============================================================================ */

      const messageInput = document.getElementById("messageBody");
      const charCountDisplay = document.getElementById("charCount");
      const smsPartsDisplay = document.getElementById("smsParts");
      const username          = sessionStorage.getItem("username");

      /* ============================================================================
          1. INITIAL USER DATA FETCH
      ============================================================================ */
      fetch(`http://localhost/rise-sms/api/v_1/user/get_user_details?username=${encodeURIComponent(username)}`, {
        method: 'GET',
        headers: { 'Authorization': username }
      })
        .then(res => res.json())
        .then(data => {
          if (data.success) {
            displayUserDetails(data.user);
          }
        })
        .catch(err => console.error("❌ User data fetch error:", err));

      /**
       * Display main user details on UI.
       */
      function displayUserDetails(user) {
        const firstName = user.name.split(" ")[0];
        document.getElementById("customer_name").innerText      = firstName;

      }

      function openLoader() {
        document.getElementById('loaderGIF').style.display = 'flex';
      }
      
      function closeLoader() {
          document.getElementById('loaderGIF').style.display = 'none';
      }

    



      function loadSenderIDs() {
        fetch("http://localhost/rise-sms/api/v_1/sms/get_sender_ids", {
          method: 'GET',
          credentials: 'include' // important if session is used
        })
        .then(res => res.json())
        .then(data => {
          if (data.success) {
            const select = document.getElementById("senderId");
            select.innerHTML = ""; // Clear existing
      
            data.senders.forEach(sid => {
              const option = document.createElement("option");
              option.value = sid.id;
              option.style.position = "absolute";
              option.textContent = sid.sender_id;
              select.appendChild(option);
            });
      
          } else {
            console.error("⚠️ Could not fetch sender IDs:", data.message);
          }
        })
        .catch(err => {
          console.error("❌ Sender ID load error:", err);
        });
      }
      
      // Call on page load
      loadSenderIDs();


      function getDoualaLocalISOTime() {
        const now = new Date();
      
        const options = {
          timeZone: 'Africa/Douala',
          year: 'numeric',
          month: '2-digit',
          day: '2-digit',
          hour: '2-digit',
          minute: '2-digit',
          hour12: false
        };
      
        const formatter = new Intl.DateTimeFormat('en-CA', options);
        const parts = formatter.formatToParts(now).reduce((acc, part) => {
          acc[part.type] = part.value;
          return acc;
        }, {});
      
        return `${parts.year}-${parts.month}-${parts.day}T${parts.hour}:${parts.minute}`;
      }
      
      // Apply to input
      const scheduleInput = document.getElementById('scheduleMessage');
      const currentTime = getDoualaLocalISOTime();
      scheduleInput.value = currentTime;
      scheduleInput.min = currentTime;

      scheduleInput.addEventListener("change", function () {
        if (scheduleInput.value < scheduleInput.min) {
          const error = document.getElementById("errorLogs") ;
          error.style.display = "block" ;
          error.innerHTML = "⛔ You cannot select a time in the past.";
          scheduleInput.value = scheduleInput.min;
        }
      });


      
      messageInput.addEventListener("input", function () {
        const charCount = messageInput.value.length;
        const parts = Math.ceil(charCount / 150) || 1;

        charCountDisplay.textContent = charCount;
        smsPartsDisplay.textContent = parts;
      });

      
      


      let buffer = "";
      let phoneNumbers = [];
      
      const receipiantInput = document.getElementById('receipiant');
      const ptags = document.getElementById('phoneTags');
      const receipiantGroup = receipiantInput.parentElement; // To insert tags into same visual block
      
      // Create a container for tags inside the input wrapper
      const tagContainer = document.createElement('div');
      tagContainer.className = 'tag-container';
      tagContainer.style.display = 'flex';
      tagContainer.style.flexWrap = 'wrap';
      tagContainer.style.alignItems = 'center';
      receipiantGroup.insertBefore(tagContainer, ptags);
      
      // Typing logic
      receipiantInput.addEventListener('input', (e) => {
        let input = e.target.value;
        let digits = input.replace(/\D/g, '');
      
        // Keep looping while we have at least one 9-digit number
        while (digits.length >= 9) {
          const phone = digits.slice(0, 9);
      
          if (!phoneNumbers.includes(phone)) {
            buffer += phone + ",";
            phoneNumbers.push(phone);
            createPhoneTag(phone);
          }
      
          digits = digits.slice(9); // <-- Correct: reduce digits for next loop
        }
      
        // Show leftover digits (incomplete phone number) back in the input
        e.target.value = digits;
      });
      
      // Create tag
      function createPhoneTag(phone) {
        const tag = document.createElement('span');
        tag.className = 'phone-tag';
        tag.textContent = phone;
      
        const close = document.createElement('span');
        close.textContent = '×';
        close.style.marginLeft = '6px';
        close.style.cursor = 'pointer';
        close.onclick = () => {
          tag.remove();
          phoneNumbers = phoneNumbers.filter(p => p !== phone);
        };
      
        tag.appendChild(close);
        tagContainer.appendChild(tag);
      }

      
      
      async function saveMessage() {
        openLoader()

        const recipient = buffer;
        const senderId = document.getElementById("senderId").value;
        const message = document.getElementById("messageBody").value.trim();
        const schedule = document.getElementById("scheduleMessage").value;
        const user_id = username ;
      
        if (!recipient || !senderId || !message) {
          alert("All fields are required.");
          return;
        }
      
        const payload = { recipient, senderId, message, schedule, user_id };
      
        try {
          const res = await fetch("http://localhost/rise-sms/api/v_1/sms/send_single_sms", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(payload)
          });
      
          const data = await res.json();
      
          if (data.success) {
            closeLoader()
            document.getElementById("scheduleSuccess").innerHTML = "Message Scheduled successfully" ;
          } else {
            alert("❌ " + data.message);
            closeLoader()
          }
      
        } catch (err) {
          closeLoader()
          console.error("❌ Send SMS error:", err);
          alert("❌ Network error");
        }
      }


      function switchTab(tabName) {
        document.querySelectorAll('.tab').forEach(tab => tab.classList.remove('active'));
        document.getElementById('current-loan').classList.remove('active');
        document.getElementById('history-loan').classList.remove('active');

        if (tabName === 'current') {
          document.querySelector('.tab:nth-child(1)').classList.add('active');
        } else {
          document.querySelector('.tab:nth-child(2)').classList.add('active');
          fetchSMSQueues() // auto fetch sms queue when this tap is clicked
        }

        document.getElementById(`${tabName}-loan`).classList.add('active');
      }
      



      async function fetchSMSQueues() {
        openLoader() // open loader

        try {
          const response = await fetch("http://localhost/rise-sms/api/v_1/sms/fetch_queue_sms");
          const result = await response.json();

          if (result.success) {

            closeLoader() // close loader at success fetching

            const tbody = document.getElementById("smsqueues");
            tbody.innerHTML = "";

            result.queues.forEach(q => {
              const row = document.createElement("tr");
              row.innerHTML = `
                <td>${formatReadableDate(q.schedule_time)}</td>
                <td>${q.recipients}</td>
                <td><span class="badge ${getStatusClass(q.status)}">${capitalize(q.status)}</span></td>
                <td>${q.message}</td>
                <td>
                  <span class="mr-4 view-all-players fs-17 ${getStatusClass('pending')}" id="delete-${q.id}"><i class="fa-solid fa-trash"></i></span>
                  <span class="mr-4 view-all-players fs-17 edit-btn" data-id="${q.id}"><i class="fa-solid fa-file-pen"></i></span>
                </td>
              `;
              tbody.appendChild(row);
              const deleteBtn = row.querySelector(`#delete-${q.id}`);
                deleteBtn.addEventListener("click", () => {
                openConfirmModal(q.id, row);
              });

               // Edit button
              const editBtn = row.querySelector(".edit-btn");
              editBtn.addEventListener("click", () => editMessage(q.id));

            });
          } else {
            alert("❌ " + result.error);
            closeLoader() // close loader at failure
          }

        } catch (err) {
          alert("Could not load sms queues.");
          closeLoader() // close loader at failure
        }
      }

      function getStatusClass(status) {
        switch (status) {
          case "completed": return "badge-success";
          case "pending": return "badge-warning";
          case "failed": return "badge-danger";
          case "cancelled": return "badge-secondary";
          default: return "badge-light";
        }
      }

      function capitalize(str) {
        return str.charAt(0).toUpperCase() + str.slice(1);
      }

      function getOrdinal(n) {
        const s = ["th", "st", "nd", "rd"];
        const v = n % 100;
        return n + (s[(v - 20) % 10] || s[v] || s[0]);
      }
      
      function formatReadableDate(dateStr) {
        const date = new Date(dateStr.replace(' ', 'T')); // Ensure valid ISO format
        const weekday = date.toLocaleDateString('en-US', { weekday: 'long' });
        const month = date.toLocaleDateString('en-US', { month: 'long' });
        const day = getOrdinal(date.getDate());
        const time = date.toLocaleTimeString('en-US', {
          hour: 'numeric',
          minute: '2-digit',
          hour12: true
        });
      
        return `${weekday} ${month} ${day} ${time}`;
      }


      function editMessage(id) {
        // You can debug here to see if it gets called
        console.log("Editing ID:", id);
        window.location.href = `edit_sms?id=${id}`;
      }
      

      let smsIdToDelete = null;
      let rowToDelete = null;

      // Open confirmation modal
      function openConfirmModal(smsId, rowElement) {
        smsIdToDelete = smsId;
        rowToDelete = rowElement;
        document.getElementById("confirmModal").style.display = "flex";
      }

      // Close modal
      function closeConfirmModal() {
        smsIdToDelete = null;
        rowToDelete = null;
        document.getElementById("confirmModal").style.display = "none";
      }

      // Confirm delete
      document.getElementById("confirmDeleteBtn").addEventListener("click", async () => {
        if (!smsIdToDelete) return;

        try {
          const response = await fetch(`http://localhost/rise-sms/api/v_1/sms/delete_queue_sms?id=${smsIdToDelete}`, {
            method: "DELETE"
          });
          const result = await response.json();

          if (result.success) {
            rowToDelete.remove(); // remove from DOM
          } else {
            alert("❌ " + result.error);
          }
        } catch (error) {
          alert("❌ Failed to delete SMS queue.");
        } finally {
          closeConfirmModal();
        }
      });

      // Cancel delete
      document.getElementById("cancelDeleteBtn").addEventListener("click", closeConfirmModal);

