// ================================
// DOM ELEMENTS AND GLOBAL VARS
// ================================
const smsSlider = document.getElementById('smsQuantitySlider');
const uniqueAmountDisplay = document.getElementById('uniqueAmountDisplay');
const pricePerSMSDisplay = document.getElementById('pricePerSMS');
const vatAmountDisplay = document.getElementById('vatAmount');
const amountPayableDisplay = document.getElementById('amountPayable');
const username = sessionStorage.getItem("username"); // Retrieve username from session

// ================================
// PRICING CONSTANTS
// ================================
const baseUnit = 10000;
const step = 2000;
const discountPerStep = 0.5;
const basePricePerSMS = 12;
const minPricePerSMS = 9.5;
const vatRate = 0.1925;

let totalPrice; // Holds final calculated amount

// ================================
// PRICE CALCULATION & DISPLAY
// ================================
function updatePricing() {
  const quantity = parseInt(smsSlider.value);
  uniqueAmountDisplay.textContent = quantity.toLocaleString();

  const extraUnits = Math.floor((quantity - baseUnit) / step);
  let pricePerSMS = basePricePerSMS - (extraUnits * discountPerStep);
  pricePerSMS = Math.max(minPricePerSMS, pricePerSMS); // Apply floor limit

  const vatAmount = pricePerSMS * vatRate;
  totalPrice = (pricePerSMS + vatAmount) * quantity;

  // Update UI values
  pricePerSMSDisplay.textContent = pricePerSMS.toFixed(1);
  vatAmountDisplay.textContent = vatAmount.toFixed(2);
  amountPayableDisplay.textContent = totalPrice.toLocaleString(undefined, { maximumFractionDigits: 0 });
}

// Initialize display on load
updatePricing();

// Recalculate pricing when slider changes
smsSlider.addEventListener('input', updatePricing);

// ================================
// USER DATA FETCH
// ================================
fetch(`http://localhost/rise-sms/api/v_1/user/get_user_details?username=${encodeURIComponent(username)}`, {
  method: 'GET',
  headers: { 'Authorization': username }
})
  .then(res => res.json())
  .then(data => {
    if (data.success) displayUserDetails(data.user);
  })
  .catch(err => console.error("❌ User data fetch error:", err));

// Display user's basic info
function displayUserDetails(user) {
  const firstName = user.name.split(" ")[0];
  document.getElementById("customer_name").innerText = firstName;
  document.getElementById("loginUserId").value = user.id;
}

// ================================
// IPHONE MODAL CONTROL
// ================================
function openIphoneAlertModal() {
  document.getElementById('iphoneiPhoneAlertModal').style.display = 'flex';
}

function closeIphoneAlertModal() {
  document.getElementById('iphoneiPhoneAlertModal').style.display = 'none';
}

function closeSuccessDiv() {
  document.getElementById('successDiv').style.display = 'none';
}

// ================================
// PAYMENT REQUEST + POLLING
// ================================
async function requestPayment() {
  const amount = parseInt(totalPrice); // Total price with VAT
  const sms_credit = smsSlider.value;
  const phone = document.getElementById('paymentNumber').value.trim();
  const user_id = document.getElementById('loginUserId').value.trim();

  openIphoneAlertModal(); // Show loading UI

  try {
    // Trigger payment via backend
    const response = await fetch("http://localhost/rise-sms/api/v_1/transactions/trigger_payments", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify({ amount, phone, user_id, sms_credit })
    });

    const result = await response.json();

    if (result.status === "success") {
      const reference = result.reference;
      let attempt = 0;
      const pollInterval = 5000; // 5 seconds
      const maxAttempts = 12;    // Max 1 minute

      // Start polling for payment status update
      const pollForStatus = setInterval(async () => {
        attempt++;

        try {
          const pollRes = await fetch(`http://localhost/rise-sms/api/v_1/transactions/check_payment_status?reference=${reference}`);
          const pollData = await pollRes.json();

          if (pollData.success) {
            const status = pollData.status.toLowerCase();

            if (status === "completed" || status === "success") {
              clearInterval(pollForStatus);
              closeIphoneAlertModal();
              document.getElementById('successDiv').style.display = "flex";
              showToast("✅ Payment completed successfully!");
            } else if (status === "failed" || status === "cancelled") {
              clearInterval(pollForStatus);
              closeIphoneAlertModal();
              alert("❌ Payment failed. Please try again.");
            }
          } else {
            console.warn("⚠️ Status check failed:", pollData.message);
          }

        } catch (err) {
          console.error("🔁 Polling error:", err);
        }

        // Stop polling after timeout
        if (attempt >= maxAttempts) {
          clearInterval(pollForStatus);
          closeIphoneAlertModal();
          document.getElementById('paymentStatus').innerText = "⏰ Payment confirmation timed out.";
        }

      }, pollInterval);

    } else {
      // Handle failure to initiate payment
      closeIphoneAlertModal();
      alert("❌ Payment initiation failed: " + (result.message || "Unknown error"));
    }

  } catch (err) {
    // Handle general request failure
    closeIphoneAlertModal();
    console.error("💥 Payment error:", err);
    alert("❌ Network error while initiating payment.");
  }
}
