      /* ============================================================================
          GLOBAL VARIABLES & DOM ELEMENTS
      ============================================================================ */
      const iphoneModal       = document.getElementById('bottomModal');
      const backdrop          = document.querySelector('.modal-backdrop');
      const openApiKey        = document.getElementById('api-key-modal');
      const apiToggleBtn      = document.getElementById("apiToggleBtn");
      const username          = sessionStorage.getItem("username");

      /* ============================================================================
          1. INITIAL USER DATA FETCH
      ============================================================================ */
      fetch(`http://localhost/rise-sms/api/v_1/user/get_user_details?username=${encodeURIComponent(username)}`, {
        method: 'GET',
        headers: { 'Authorization': username }
      })
        .then(res => res.json())
        .then(data => {
          if (data.success) {
            displayUserDetails(data.user);
            renderSenderIds(data.sender_ids);
          }
        })
        .catch(err => console.error("❌ User data fetch error:", err));

      /**
       * Display main user details on UI.
       */
      function displayUserDetails(user) {
        const firstName = user.name.split(" ")[0];
        document.getElementById("customer_name").innerText      = firstName;
        document.getElementById("loggedin_username").innerText  = firstName;
        document.getElementById("loginUserId").value            = user.id;
        document.getElementById("currentApiKey").innerText      = user.api_key;
        document.getElementById("sms_balance").innerText        = Number(user.sms_balance).toLocaleString() + " SMS";

        apiToggleBtn.setAttribute("aria-pressed", user.is_api_active ? "true" : "false");
      }

      /**
       * Render sender IDs for the current user.
       */
      function renderSenderIds(senderIds) {
        const container = document.getElementById("AllSenderIds");
        container.innerHTML = "";

        senderIds.forEach(sender => {
          const initials = sender.sender_id.slice(0, 2).toUpperCase();
          const statusColor = sender.is_white_listed == 1 ? "bg-success" : "bg-danger";
          const statusText = sender.is_white_listed == 1 ? "Whitelisted" : "Pending...";
          const otpStatus = sender.is_otp_whitelisted == 1 ? "OTP On" : "OTP Off";

          const html = `
            <div class="flex-details">
              <div class="d-flex align-items-center gap-2">
                <span class="avatar rounded-circle bg-gradient-in mr-2">
                  <span class="id_caption">${initials}</span>
                </span>
                <div class="text senderid_name">${sender.sender_id}</div>
                <small class="text-2 pending-status">
                  <span class="badge badge-dot mr-4">
                    <i class="${statusColor}"></i>
                    <span class="status">${statusText}</span>
                    <small class="text-2">${otpStatus}</small>
                  </span>
                </small>
              </div>
            </div>
          `;
          container.insertAdjacentHTML("beforeend", html);
        });
      }

      /* ============================================================================
          2. API KEY TOGGLE (Enable/Disable API usage)
      ============================================================================ */
      apiToggleBtn.addEventListener("click", async () => {
        const newState = apiToggleBtn.getAttribute("aria-pressed") !== "true";
        apiToggleBtn.setAttribute("aria-pressed", newState ? "true" : "false");

        try {
          const res = await fetch("http://localhost/rise-sms/api/v_1/user/toggle_api_status", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({
              username: username,
              is_api_active: newState ? 1 : 0
            })
          });

          const data = await res.json();
          if (!data.success) showToast("❌ Failed to update API status.");
        } catch (err) {
          console.error("API toggle error:", err);
          showToast("❌ Network error while updating API status.");
        }
      });

      /* ============================================================================
          3. API KEY COPY & REGENERATION
      ============================================================================ */
      function copyApiKey() {
        const apiKey = document.getElementById('currentApiKey').textContent;
        navigator.clipboard.writeText(apiKey)
          .then(() => showToast("✅ API key copied!"))
          .catch(err => {
            console.error("Clipboard error:", err);
            showToast("❌ Failed to copy.");
          });
      }

      async function regenerateApiKey() {
        if (!username) return alert("User not identified.");

        try {
          const res = await fetch("http://localhost/rise-sms/api/v_1/authenticate/regenerate_api_key", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ username })
          });

          const data = await res.json();

          if (data.success && data.new_api_key) {
            document.getElementById("currentApiKey").innerText = data.new_api_key;
            showToast("🔄 API key regenerated successfully!");
          } else {
            showToast("❌ Failed to regenerate API key.");
          }
        } catch (error) {
          console.error("Regenerate error:", error);
          alert("❌ Server error while regenerating API key.");
        }
      }

      /* ============================================================================
          4. MODAL CONTROLS (Bottom Modal & API Key Modal)
      ============================================================================ */
      function openIphoneModal() {
        iphoneModal.classList.add('show');
        backdrop.classList.add('show');
      }

      function closeIphoneModal() {
        iphoneModal.classList.remove('show');
        backdrop.classList.remove('show');
      }

      function openAPIKey() {
        openApiKey.style.display = "flex";
      }

      function closeApiKeyModal() {
        openApiKey.style.display = "none";
      }

      /* ============================================================================
          5. ADD NEW SENDER ID
      ============================================================================ */
      async function saveSenderId() {
        const senderIdInput = document.getElementById("senderId");
        const sampleMessageInput = document.getElementById("sampleMessage");
        const userId = document.getElementById("loginUserId").value.trim();
        const senderId = senderIdInput.value.trim();
        const message = sampleMessageInput.value.trim();
        const errorLog = document.getElementById("errorLogs");

        // Clear previous styles
        senderIdInput.style.border = "";
        sampleMessageInput.style.border = "";
        errorLog.style.display = "none";

        // === Validation
        if (!senderId || !message || senderId.length > 11) {
          errorLog.innerHTML = senderId.length > 11
            ? "❌ Sender ID must not exceed 11 characters."
            : "❌ Please fill in all required fields.";
          errorLog.style.display = "block";

          if (!senderId) senderIdInput.style.border = "2px solid red";
          if (!message) sampleMessageInput.style.border = "2px solid red";
          return;
        }

        try {
          const res = await fetch("http://localhost/rise-sms/api/v_1/add_sender_id", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify({ sender_id: senderId, sample_message: message, user_id: userId })
          });

          const result = await res.json();

          if (res.ok && result.id) {
            appendSenderIdToList(senderId, result.id);
            senderIdInput.value = "";
            sampleMessageInput.value = "";
          } else {
            errorLog.innerHTML = "❌ Failed to add sender ID.";
            errorLog.style.display = "block";
          }

        } catch (error) {
          errorLog.innerHTML = "❌ Network error.";
          errorLog.style.display = "block";
        }
      }

      /**
       * Append new sender ID to UI after saving.
       */
      function appendSenderIdToList(senderId, recordId) {
        const container = document.getElementById("senderIds");
        const initials = senderId.slice(0, 2).toUpperCase();

        const html = `
          <div class="flex-details" id="senderIdRow-${recordId}">
            <div class="d-flex align-items-center gap-2">
              <span class="avatar rounded-circle bg-gradient-in mr-2">
                <span class="id_caption">${initials}</span>
              </span>
              <div class="text senderid_name">${senderId}</div>
              <small class="text-2 pending-status">
                <span class="badge badge-dot mr-4">
                  <i class="bg-danger"></i>
                  <span class="status">Pending...</span>
                  <small class="text-2">OTP Off</small>
                </span>
              </small>
            </div>
          </div>
        `;
        container.insertAdjacentHTML("beforeend", html);
      }

      /* ============================================================================
          6. TOAST HELPER
      ============================================================================ */
      function showToast(message) {
        const toast = document.createElement("div");
        toast.innerText = message;
        Object.assign(toast.style, {
          position: "fixed", bottom: "30px", left: "50%",
          transform: "translateX(-50%)", padding: "12px 20px",
          background: "#333", color: "#fff", borderRadius: "6px",
          zIndex: 9999, fontSize: "14px", boxShadow: "0 2px 6px rgba(0,0,0,0.2)"
        });
        document.body.appendChild(toast);
        setTimeout(() => toast.remove(), 2500);
      }

      /* ============================================================================
          7. TAB SWITCHING
      ============================================================================ */
      function switchTab(tabName) {
        document.querySelectorAll('.tab').forEach(tab => tab.classList.remove('active'));
        document.getElementById('current-loan').classList.remove('active');
        document.getElementById('history-loan').classList.remove('active');

        if (tabName === 'current') {
          document.querySelector('.tab:nth-child(1)').classList.add('active');
        } else {
          fetchSMSLogs() // auto fetch sms logs when this tap is clicked
          document.querySelector('.tab:nth-child(2)').classList.add('active');
        }

        document.getElementById(`${tabName}-loan`).classList.add('active');
      }

      /* ============================================================================
          8. FETCH & DISPLAY TRANSACTIONS
      ============================================================================ */
      async function fetchTransactions() {
        try {
          const response = await fetch("http://localhost/rise-sms/api/v_1/transactions/get_sms_transactions");
          const result = await response.json();

          if (result.success) {
            const tbody = document.getElementById("transactionTableBody");
            tbody.innerHTML = "";

            result.transactions.forEach(tx => {
              const row = document.createElement("tr");
              row.innerHTML = `
                <td>${new Date(tx.created_at).toLocaleString()}</td>
                <td>${parseFloat(tx.amount).toLocaleString()} ${tx.currency || 'XAF'}</td>
                <td><span class="badge ${getStatusClass(tx.status)}">${capitalize(tx.status)}</span></td>
                <td>${capitalize(tx.transaction_type)}</td>
                <td>${tx.sms_credit || '-'}</td>
              `;
              tbody.appendChild(row);
            });
          } else {
            alert("❌ " + result.error);
          }

        } catch (err) {
          console.error("❌ Transaction fetch error:", err);
          alert("Could not load transactions.");
        }
      }

      function getStatusClass(status) {
        switch (status) {
          case "completed": return "badge-success";
          case "pending": return "badge-warning";
          case "failed": return "badge-danger";
          case "cancelled": return "badge-secondary";
          default: return "badge-light";
        }
      }

      function capitalize(str) {
        return str.charAt(0).toUpperCase() + str.slice(1);
      }

      // Initialize transaction load on page load
      fetchTransactions();